package com.horstmann;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import io.smallrye.common.annotation.RunOnVirtualThread;
import jakarta.ws.rs.GET;
import jakarta.ws.rs.Path;
import jakarta.ws.rs.Produces;
import jakarta.ws.rs.core.MediaType;

import java.lang.reflect.AccessibleObject;
import java.lang.reflect.Array;
import java.lang.reflect.Field;
import java.lang.reflect.Modifier;
import java.util.ArrayList;

@Path("/threadlocals")
public class ThreadLocals {
    @RunOnVirtualThread
    @Path("/virtual")
    @GET
    @Produces(MediaType.TEXT_PLAIN)
    public String virtual() throws Exception {
        return threadLocals(false);
    }

    @Path("/platform")
    @GET
    @Produces(MediaType.TEXT_PLAIN)
    public String platform() throws Exception {
        return threadLocals(false);
    }
    
    @RunOnVirtualThread
    @Path("/virtual/verbose")
    @GET
    @Produces(MediaType.TEXT_PLAIN)
    public String virtualVerbose() throws Exception {
        return threadLocals(true);
    }

    @Path("/platform/verbose")
    @GET
    @Produces(MediaType.TEXT_PLAIN)
    public String platformVerbose() throws Exception {
        return threadLocals(true);
    }

    private static final ThreadLocal<ObjectMapper> threadLocalObjectMapper
            = ThreadLocal.withInitial(ObjectMapper::new);

    private static final ScopedValue<ObjectMapper> scopedValueObjectMapper
            = ScopedValue.newInstance();

    @Path("/platform/props")
    @GET
    @Produces(MediaType.APPLICATION_JSON)
    public String platformProps() throws Exception {
        return taskWithThreadLocal();
    }

    @RunOnVirtualThread
    @Path("/virtual/props")
    @GET
    @Produces(MediaType.APPLICATION_JSON)
    public String virtualProps() throws Exception {
        return taskWithThreadLocal();
    }

    @RunOnVirtualThread
    @Path("/scoped/props")
    @GET
    @Produces(MediaType.APPLICATION_JSON)
    public String scopedProps() throws Exception {
        return ScopedValue
                .where(scopedValueObjectMapper, new ObjectMapper())
                .call(() -> taskWithScopedValue());
    }

    String taskWithThreadLocal() throws JsonProcessingException {
        ObjectMapper mapper = threadLocalObjectMapper.get();
        return mapper.writeValueAsString(System.getProperties());
    }

    String taskWithScopedValue() throws JsonProcessingException {
        ObjectMapper mapper = scopedValueObjectMapper.get();
        return mapper.writeValueAsString(System.getProperties());
    }

    // http://blog.igorminar.com/2009/03/identifying-threadlocal-memory-leaks-in.html
    public static String threadLocals(boolean verbose) throws ReflectiveOperationException {
        return "Thread locals:\n" +
            threadLocalsMap("threadLocals", verbose) +
            "Inheritable thread locals:\n"  +
            threadLocalsMap("inheritableThreadLocals", verbose) ;
    }
    
    public static String threadLocalsMap(String fieldName, boolean verbose) throws ReflectiveOperationException {
        Thread thread = Thread.currentThread();
        Field threadLocalsField = Thread.class.getDeclaredField(fieldName);
        threadLocalsField.setAccessible(true);
        Class<?> threadLocalMapKlazz = Class
            .forName("java.lang.ThreadLocal$ThreadLocalMap");
        Field tableField = threadLocalMapKlazz.getDeclaredField("table");
        tableField.setAccessible(true);

        Object threadLocals = threadLocalsField.get(thread);
        if (threadLocals == null) return "none\n";
        Object table = tableField.get(threadLocals);

        int threadLocalCount = Array.getLength(table);
        StringBuilder classSb = new StringBuilder();

        for (int i = 0; i < threadLocalCount; i++) {
            Object entry = Array.get(table, i);
            if (entry != null) {
                Field valueField = entry.getClass().getDeclaredField("value");
                valueField.setAccessible(true);
                Object value = valueField.get(entry);
                if (verbose) {
                    classSb.append(new ObjectAnalyzer().toString(value)).append("\n");
                } else if (value != null) {
                  classSb.append(value.getClass().getName()).append("\n");
                }
                else {
                  classSb.append("null\n");
                }                
            }
        }
        return classSb.toString();
    }
}

// Core Java vol 1 ch 5
class ObjectAnalyzer {
    private ArrayList<Object> visited = new ArrayList<>();

    /**
     * Converts an object to a string representation that lists all fields.
     * @param obj an object
     * @return a string with the object's class name and all field names and values
     */
    public String toString(Object obj)
        throws ReflectiveOperationException {
        if (obj == null) return "null";
        if (visited.contains(obj)) return "...";
        visited.add(obj);
        Class<?> cl = obj.getClass();
        if (cl == String.class) return (String) obj;
        if (cl.isArray()) {
            int len = Array.getLength(obj);
            String r = cl.getComponentType().getName() + "[" + len + "]{";
            final int LIMIT = 10;
            for (int i = 0; i < Math.min(len, LIMIT); i++) {
                if (i > 0) r += ",";
                Object val = Array.get(obj, i);
                if (cl.getComponentType().isPrimitive()) r += val;
                else r += toString(val);
            }
            if (len > LIMIT) r += ",...";
            return r + "}";
        }

        String r = cl.getName();
        // inspect the fields of this class and all superclasses
        r += "[";
        do {
            Field[] fields = cl.getDeclaredFields();
            try {
                AccessibleObject.setAccessible(fields, true);
                // get the names and values of all fields
                for (Field f : fields) {
                    if (!Modifier.isStatic(f.getModifiers())) {
                        if (!r.endsWith("[")) r += ",";
                        r += f.getName() + "=";
                        Class<?> t = f.getType();
                        Object val = f.get(obj);
                        if (t.isPrimitive()) r += val;
                        else {
                            try {
                                r += toString(val);
                            } catch (Exception ex) {
                                r += "?";
                            }
                        }
                    }
                }
            } catch (Exception ex) {
                if (!r.endsWith("[")) r += ",";
                r += "?";
            }
            cl = cl.getSuperclass();
        }
        while (cl != null);
        r += "]";

        return r;
    }
}