package com.horstmann;

import io.smallrye.common.annotation.RunOnVirtualThread;
import jakarta.ws.rs.GET;
import jakarta.ws.rs.Path;
import jakarta.ws.rs.PathParam;
import jakarta.ws.rs.Produces;
import jakarta.ws.rs.core.MediaType;
import org.eclipse.microprofile.config.ConfigProvider;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.nio.file.*;
import java.nio.file.attribute.BasicFileAttributes;
import java.util.Map;
import java.util.concurrent.StructuredTaskScope;
import java.util.stream.Collectors;
import java.util.stream.Gatherers;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;

@Path("/search")
public class Search {
    String zipname = ConfigProvider.getConfig().getOptionalValue("java.src.zip", String.class).orElse("");
    String root = ConfigProvider.getConfig().getOptionalValue("java.src.dir", String.class).orElse("");

    @RunOnVirtualThread
    @Path("/virtual/{key}")
    @GET
    @Produces(MediaType.TEXT_PLAIN)
    public String searchVirtual(@PathParam("key") String key) throws Exception {
        return search(key);
    }

    @Path("/platform/{key}")
    @GET
    @Produces(MediaType.TEXT_PLAIN)
    public String searchPlatform(@PathParam("key") String key) throws Exception {
        return search(key);
    }

    private String search(String key) throws IOException {
        StringBuilder result = new StringBuilder();
        FileSystem fs = FileSystems.newFileSystem(java.nio.file.Path.of(zipname));
        return Files.walk(fs.getPath("/"))
                .filter(Files::isRegularFile)
                .filter(f -> {
                    try {
                        return Files.readString(f).contains(key);
                    } catch (IOException __) { return false; }
                })
                .map(f -> f.toString())
                .collect(Collectors.joining("\n"));
    }

    @RunOnVirtualThread
    @Path("/concurrent/{key}")
    @GET
    @Produces(MediaType.TEXT_PLAIN)
    public String searchConcurrent(@PathParam("key") String key) throws Exception {
        return Files.walk(java.nio.file.Path.of(root))
            .filter(Files::isRegularFile)
            .gather(Gatherers.mapConcurrent(100, f -> {
                try {
                    return Map.entry(f, Files.readString(f));
                } catch (IOException _) { return null; }
            }))
            .filter(p -> p != null && p.getValue().contains(key))
            .map(f -> f.getKey().toString())
            .collect(Collectors.joining("\n"));
    }
}
