package com.horstmann;

import io.smallrye.common.annotation.RunOnVirtualThread;
import jakarta.ws.rs.GET;
import jakarta.ws.rs.Path;
import jakarta.ws.rs.Produces;
import jakarta.ws.rs.WebApplicationException;
import jakarta.ws.rs.core.StreamingOutput;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URI;
import java.net.http.HttpClient;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.time.Duration;
import java.time.LocalDate;
import java.time.format.DateTimeFormatter;
import java.time.temporal.ChronoUnit;
import java.util.NoSuchElementException;
import java.util.concurrent.Semaphore;
import java.util.random.RandomGenerator;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

@Path("/image")
public class Image {
    private final HttpClient client = HttpClient.newHttpClient();
    private static Semaphore sem = new Semaphore(20);

    @RunOnVirtualThread
    @Path("/1")
    @GET
    @Produces("image/jpeg")
    public StreamingOutput random1() throws Exception {        
        byte[] image = randomImage();
        return new StreamingOutput() {
            public void write(OutputStream output)
                throws IOException, WebApplicationException {
                output.write(image);
                output.close();
            }
        };        
    }

    @RunOnVirtualThread
    @Path("/2")
    @GET
    @Produces("image/jpeg")
    public StreamingOutput random2() throws Exception {
        sem.acquire();
        try {
            byte[] image = randomImage();
            return new StreamingOutput() {
                public void write(OutputStream output)
                    throws IOException, WebApplicationException {
                    output.write(image);
                    output.close();
                }
            };        
        } finally {
            sem.release();
        }
    }
    
    private byte[] randomImage() throws IOException, InterruptedException {
        String url = imageURL(randomDate(LocalDate.of(2005, 3, 1)));
        HttpRequest request = HttpRequest.newBuilder()
            .GET()
            .uri(URI.create(url))
            .timeout(Duration.ofSeconds(60))
            .build();
        HttpResponse<String> response = client.send(request, HttpResponse.BodyHandlers.ofString());
        String body = response.body();
        String url2 = findImage(body);
        if (url2 == null) {
            throw new NoSuchElementException("No image found in " + url);
        }
        HttpRequest request2 = HttpRequest.newBuilder()
            .GET()
            .uri(URI.create(url2))
            .timeout(Duration.ofSeconds(60))
            .build();
        HttpResponse<InputStream> response2 = client.send(request2,  HttpResponse.BodyHandlers.ofInputStream());
        return response2.body().readAllBytes();
    }


    private String findImage(String body) {
        Pattern pattern = Pattern
            .compile("src=\"(?<src>https://upload.wikimedia.org/[^\"]+)\"");
        Matcher matcher = pattern.matcher(body);
        if (matcher.find()) {
            return matcher.group("src");
        }
        else {
            return null;
        }
    }

    private LocalDate randomDate(LocalDate start) {
        LocalDate now = LocalDate.now();
        long daysBetween = ChronoUnit.DAYS.between(start, now);
        return start.plusDays(RandomGenerator.getDefault().nextLong(daysBetween));
    }
    
    private String imageURL(LocalDate date) {
        return "https://commons.wikimedia.org/wiki/Special:FeedItem/potd/"
            + DateTimeFormatter.BASIC_ISO_DATE.format(date) + "000000/en";
    }
}
